%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATLAB code: Exercise 2.12(b)
% File: nlm_girf.m
%
% Program to compute GI response functions (GIRF). 
% Based on Koop et al. (1996), J. of Econometrics, 74(1), 119-147.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function nlm_girf( )
    clear all % clear variables from memory 
    clc       % clear command window   
    rng(123)  % seeds the random number generator  
    % Parameters
    delta = 1;           % Size of the shock     
    phi1  = 0.9;         % AR(1), and first TAR parameter               
    phi2  = -0.50;       % second TAR parameter                    
    phi   = phi1+ phi2;
    B     = 1000;        % Number of Bootstrap replications       
    H     = 10;          % Maximum forecast horizon of impulses     
    % Simulate the data   
    e = randn(B,1);
    y = zeros(B,1);    
    for i = 2:B
        y(i) = phi1*y(i-1) + phi2*y(i-1)*(y(i-1)<=0) + e(i);
    end
       
    % Total number of draws needed for a given initial history  
    % is B-1, and H+1 for the number of impulse horizons.       
    BH      = (B-1)*(H+1);                                             
    impulse = zeros(B-1,H+1);

    % Loop through the data to change the 
    % initial history (condition)   
    for i=1:B-1      
        %  Bootstrap residuals 
        ind    = fix(rand(BH,1)*(B-1) + 1 );
        e_boot = e(ind);              
        e_boot = reshape(e_boot,H+1,B-1);
        ye0    = zeros(H+1,B-1);
        ye1    = zeros(H+1,B-1);
    
        % Loop through horizon of impulse responses
        for j = 2:B-1                                                                
            % Initial condition based on a bootstrap draw              
            ye0(:,j) = model(e_boot(:,j),e_boot(1,j),phi1,phi2);                   
            % Initial condition based on history (i subscript) 
            % plus shock delta  
            ye1(:,j) = model(e_boot(:,j),e(i)+delta,phi1,phi2);           
        end
        % Average over horizon given an initial condition          
        impulse(i,:) = (mean(ye1,2) - mean(ye0,2))';          
    end
    
    % Average each impulse across histories for SETAR process  
    impulse_girf = mean(impulse)';    
    
    % AR(1) Linear impulse responses    
    impulse_linear = delta*recserar(zeros(H+1,1),1,phi);   

    format short
    disp(' GI Response  AR(1) Linear ')
    disp([impulse_girf impulse_linear]);   
end
%
%--------------------------- Functions --------------------------
%  Model used to compute GI Response Function
%----------------------------------------------------------------
function y = model(e,y0,phi1,phi2)
    y = y0 + zeros(length(e),1);
    for i=2:length(y)
        % SETAR model     
        y(i) = phi1*y(i-1) + phi2*y(i-1)*(y(i-1)<=0) + e(i); 
    end
end

function result = recserar(x,y0,a)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATLAB code: Exercise 2.12(b)
% PURPOSE: computes a vector of autoregressive recursive series
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% USAGE: result = recserar(x,y0,a)
% where: x  = matrix of dimension (n,k)
%        y0 = matrix of dimension (p,k)
%        a  = matrix of dimension (p,k)
%--------------------------------------------------------------
% RETURNS:
%        results(1:n,1:k) = contains columns computed
%        recursively with result=y0 for rows 1:p, and
%        result(j,:)=result(j-1,:).*a(1:p,:) + x(j,:)
%        for rows p+1:n.
%
% written by:
% Kit Baum, Dept. of Economics, Boston College
% Chestnut Hill MA 02467 USA, baum@bc.edu
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if (nargin ~= 3)
   error('Wrong number of arguments to recserar');
end
[n1 k1] = size(x);
[p1 k2] = size(y0);
[p2 k3] = size(a);
if (k1 ~= k2)
   error('recserar x, y0 must have same number of columns');
end
if (k1 ~= k3)
   error('recserar x, a must have same number of columns');
end
if (p1 ~= p2)
   error('recserar y0, a must have same number of rows');
end
result = zeros(n1,k1);
for j=1:p1;
    result(j,:) = y0(j,:);
end
for j=(p1+1):n1;
    result(j,:) = x(j,:);
    for k=1:p1;
        result(j,:) = result(j,:)+a(k,:).*result(j-k,:);
    end
end
